import { HttpStatus, INestApplication } from '@nestjs/common';
import request from 'supertest';
import { getPrivateApp } from '../utils/setup';
import { v4 as uuidv4 } from 'uuid';

let app: INestApplication;

const offerToBeDeleted = {
  assetid: uuidv4(),
  oid: uuidv4(),
  resource: 'http://resource.jpg',
  beneficiary: '0x57fFC30a883932F98fE94Fc0eE4390C4BA5A9F2a',
  price: '10',
  cap_duration: '31536000',
  cds_target: { key: 'value' },
  cds_sl: { key: 'value' },
};
const offertNotToBeDeleted = {
  assetid: uuidv4(),
  oid: uuidv4(),
  resource: 'http://resource.jpg',
  beneficiary: '0x57fFC30a883932F98fE94Fc0eE4390C4BA5A9F2a',
  price: '10',
  cap_duration: '31536000',
  cds_target: { key: 'value' },
  cds_sl: { key: 'value' },
};

describe('Offering (e2e)', () => {
  beforeAll(async () => {
    app = await getPrivateApp();
  });

  it(`/POST /tm/v1.0/offerings - Add offer with missing properties stops addition`, async () => {
    const response = await request(app.getHttpServer())
      .post('/tm/v1.0/offerings')
      .send({
        oid: undefined,
        resource: undefined,
        beneficiary: undefined,
        price: undefined,
        cap_duration: undefined,
      });
    expect(response.statusCode).toBe(HttpStatus.INTERNAL_SERVER_ERROR);
  });

  it(`/POST /tm/v1.0/offerings - Post offer to be deleted works`, async () => {
    const response = await request(app.getHttpServer())
      .post('/tm/v1.0/offerings')
      .send(offerToBeDeleted);

    expect(response).toBeTruthy();
    expect(response.statusCode).toBe(HttpStatus.CREATED);
  });

  it(`/DELETE /tm/v1.0/offerings - Delete and check offer works `, async () => {
    const responseDelete = await request(app.getHttpServer()).delete(
      `/tm/v1.0/offerings/${offerToBeDeleted.oid}`,
    );

    expect(responseDelete).toBeTruthy();
    expect(responseDelete.statusCode).toBe(HttpStatus.OK);
    expect(responseDelete.text.length).toBe(66);

    const responseExist = await request(app.getHttpServer())
      .get('/tm/v1.0/offerings/exist')
      .query({ oid: offerToBeDeleted.oid });

    expect(responseExist).toBeTruthy();
    expect(['false']).toContain(responseExist.text);
    expect(responseExist.statusCode).toBe(HttpStatus.OK);
  });

  it(`/POST /tm/v1.0/offerings - Add offer`, async () => {
    const response = await request(app.getHttpServer())
      .post('/tm/v1.0/offerings')
      .send(offertNotToBeDeleted);

    expect(response).toBeTruthy();
    expect(response.statusCode).toBe(HttpStatus.CREATED);
  });

  it(`/GET /tm/v1.0/offerings/exist - Check if offer does exist`, async () => {
    const response = await request(app.getHttpServer())
      .get('/tm/v1.0/offerings/exist')
      .query({ oid: offertNotToBeDeleted.oid });

    expect(response).toBeTruthy();
    expect(['true']).toContain(response.text);
    expect(response.statusCode).toBe(HttpStatus.OK);
  });

  it(`/GET /tm/v1.0/offeriangs/uri - check if offer return uri correctly`, async () => {
    const response = await request(app.getHttpServer())
      .get('/tm/v1.0/offerings/uri')
      .query({ oid: offertNotToBeDeleted.oid });

    expect(response.text).toBeTruthy();
    expect(response.text).toEqual(offertNotToBeDeleted.resource);
    expect(response.statusCode).toBe(HttpStatus.OK);
  });
});
